'use strict';

var CryptoJS = require('crypto-js');
var useElementPlusTheme = require('use-element-plus-theme');
var core = require('@vueuse/core');

class SafeCommunicator {
    constructor(key) {
        this.secretKey = key || '';
    }
    encodeMessage(message) {
        const payload = JSON.stringify(message);
        return this.secretKey
            ? CryptoJS.AES.encrypt(payload, this.secretKey).toString()
            : payload;
    }
    decodeMessage(data) {
        try {
            const payload = this.secretKey
                ? CryptoJS.AES.decrypt(data, this.secretKey).toString(CryptoJS.enc.Utf8)
                : data;
            return JSON.parse(payload);
        }
        catch (e) {
            throw new Error('消息解密失败');
        }
    }
    verifyMessageSignature(data, signature) {
        // TODO: 实现消息签名验证
        return true;
    }
    generateMessageNonce() {
        // TODO: 实现消息随机数生成
        return '';
    }
}

class BridgeCore {
    constructor(config) {
        // 模块，支持注册事件
        this.modules = new Map();
        this.handleMessage = (event) => {
            if (this.strict && !this.validateOrigin(event.origin))
                return;
            try {
                // const message = this.parseMessage(event.data)
                this.routeMessage(event.data);
            }
            catch (err) {
                this.handleError('[BridgeCore] 消息处理失败:', err);
            }
        };
        this.allowedOrigins = config.allowedOrigins || [];
        this.secretKey = config.secretKey;
        this.iframeContext = config.iframeContext;
        this.tabContext = config.tabContext;
        this.source = config.source;
        this.communicator = new SafeCommunicator(this.secretKey);
        this.strict = !!config.strict;
        this.initListener();
    }
    initListener() {
        window.addEventListener('message', this.handleMessage);
    }
    parseMessage(data) {
        try {
            return this.communicator.decodeMessage(data);
        }
        catch (err) {
            throw new Error('消息解析失败');
        }
    }
    routeMessage(message) {
        try {
            if (typeof message === 'object' && message !== null && !Array.isArray(message)) {
                const handler = this.modules.get(message.type);
                handler === null || handler === void 0 ? void 0 : handler(message.payload);
            }
        }
        catch (err) {
            this.handleError('[BridgeCore] 消息处理失败:', err);
        }
    }
    // 安全验证
    validateOrigin(origin) {
        return this.allowedOrigins.includes(origin);
    }
    // 错误处理
    handleError(code, error) {
        console.error(`[CoreBridge Error ${code}]`, error);
    }
    registerHandler(type, handler) {
        if (typeof type !== 'string') {
            console.error(`[CoreBridge Error: 请使用字符串作为事件名称]`);
            return;
        }
        this.modules.set(type, handler);
    }
    // 发送信息
    send(type, payload, target) {
        var _a, _b, _c;
        // TODO：暂未使用加密，后续有加密需求时可直接启动SafeCommunicator工具对message进行加/解密
        const message = {
            type,
            payload,
        };
        try {
            if (this.source === 'container') {
                // 父传子: iframe
                (_a = this.iframeContext) === null || _a === void 0 ? void 0 : _a.forEach((context) => {
                    var _a;
                    (_a = context === null || context === void 0 ? void 0 : context.contentWindow) === null || _a === void 0 ? void 0 : _a.postMessage(message, target || '*');
                });
                // 父传子: 新tab页
                //  由于内网浏览器版本低于94.0.4586.0，导致无法读取/保存window.open的返回值，固通过Map收集每个window.open的返回值的postMessage
                (_b = this.tabContext) === null || _b === void 0 ? void 0 : _b.forEach((value, key) => {
                    value(message, key);
                });
            }
            else if (this.source === 'subApp') {
                if (window.opener) {
                    // TODO: 判断方式待优化
                    // 子传父：新tab页
                    (_c = window === null || window === void 0 ? void 0 : window.opener) === null || _c === void 0 ? void 0 : _c.postMessage(message, this.allowedOrigins[0] || '*');
                }
                else {
                    // 子传父：iframe
                    window.parent.postMessage(message, this.allowedOrigins[0] || '*');
                }
            }
        }
        catch (e) {
            throw new Error(`消息发送失败: ${e}`);
        }
    }
    destroy() {
        window.removeEventListener('message', this.handleMessage);
    }
    verifyMessageSignature(message, signature) {
        return this.communicator.verifyMessageSignature(message, signature);
    }
    generateMessageNonce() {
        return this.communicator.generateMessageNonce();
    }
    addAllowedOrigin(origin) {
        this.allowedOrigins.push(origin);
    }
    clearAllowedOrigins() {
        this.allowedOrigins = [];
    }
    // 更新Iframe主体
    updateContext(context, type) {
        if (type === 'iframe') {
            this.iframeContext = context;
        }
        else if (type === 'tab') {
            this.tabContext = context;
        }
    }
}

class ThemeModule {
    constructor(bridge, themeStoreKey, themeConfig) {
        this.bridge = bridge;
        // 初始化主题配置
        this.store = themeConfig || {
            primary: '#409EFF',
            success: '#67C23A',
            error: '#F56C6C',
            warning: '#E6A23C'
        };
        this.initElementTheme();
        this.setupHandlers();
    }
    initElementTheme() {
        const { changeTheme } = useElementPlusTheme.useElementPlusTheme();
        changeTheme(this.store.primary);
        this.applyVariables();
    }
    setupHandlers() {
        // 监听父应用发来的更新
        this.bridge.registerHandler('THEME_UPDATE', (payload) => {
            this.store = payload;
            this.applyVariables();
            this.sendAck();
        });
    }
    applyVariables() {
        const root = document.documentElement;
        Object.entries(this.store).forEach(([key, val]) => {
            // 设置全局变量
            root.style.setProperty(`--el-color-${key}`, val);
        });
    }
    // 主动更新方法（父应用调用）
    updateTheme(newTheme, targetOrigin) {
        // 合并更新并持久化
        this.store = Object.assign(Object.assign({}, this.store), newTheme);
        // 发送给子应用
        this.bridge.send('THEME_UPDATE', this.store, targetOrigin);
        this.applyVariables();
    }
    sendAck() {
        this.bridge.send('THEME_UPDATE_ACK', { success: true });
    }
}

class FontModule {
    constructor(bridge) {
        this.bridge = bridge;
        this.store = core.useStorage('font-config', {
            baseSize: 14,
            scaleFactor: 1
        });
        this.applyFontSettings();
        this.setupHandlers();
    }
    setupHandlers() {
        // 监听父应用发来的更新
        this.bridge.registerHandler('FONT_UPDATE', (payload) => {
            this.store.value = payload;
            this.applyFontSettings();
            this.sendAck();
        });
    }
    applyFontSettings() {
        const { baseSize, scaleFactor } = this.store.value;
        const root = document.documentElement;
        // 应用基础字体
        root.style.fontSize = `${baseSize}px`;
        // 应用缩放比例
        document.querySelectorAll('.el').forEach(el => {
            el.style.fontSize = `${baseSize * scaleFactor}px`;
        });
    }
    // 主动更新方法（父应用调用）
    updateFont(config) {
        // 合并更新并持久化
        this.store.value = Object.assign(Object.assign({}, this.store.value), config);
        // 发送给子应用
        this.bridge.send('FONT_UPDATE', this.store.value);
        this.applyFontSettings();
    }
    sendAck() {
        this.bridge.send('FONT_UPDATE_ACK', { success: true });
    }
}

const COOKIE_TYPE = 'cookie';
const GET_COOKIE_TYPE = 'get-top-cookie';
const FRAME_USER_INFO_TYPE = 'frame-user-info';
const FRAME_REFRESH = 'frame-refresh';
const COOKIE_NAME = 'x-customs-user';
const PARENT_OPEN = 'parent-open';

exports.BridgeCore = BridgeCore;
exports.COOKIE_NAME = COOKIE_NAME;
exports.COOKIE_TYPE = COOKIE_TYPE;
exports.FRAME_REFRESH = FRAME_REFRESH;
exports.FRAME_USER_INFO_TYPE = FRAME_USER_INFO_TYPE;
exports.FontModule = FontModule;
exports.GET_COOKIE_TYPE = GET_COOKIE_TYPE;
exports.PARENT_OPEN = PARENT_OPEN;
exports.ThemeModule = ThemeModule;
